document.addEventListener('DOMContentLoaded', async () => {
  const yearSelect = document.getElementById('year');
  const exportBtn = document.getElementById('exportBtn');
  const statusEl = document.getElementById('status');
  const statsEl = document.getElementById('stats');
  const totalEntriesEl = document.getElementById('totalEntries');

  function setStatus(message, type = 'info') {
    statusEl.textContent = message;
    statusEl.className = `status-message ${type}`;
  }

  function setLoading(loading) {
    exportBtn.disabled = loading;
    exportBtn.classList.toggle('loading', loading);
    if (loading) {
      exportBtn.querySelector('.btn-text').textContent = 'Exporting';
    } else {
      exportBtn.querySelector('.btn-text').textContent = 'Export History';
    }
  }

  // Detect available years from history
  async function detectAvailableYears() {
    setStatus('Scanning history...', 'info');
    exportBtn.disabled = true;

    try {
      // Get all history to find date range
      const results = await chrome.history.search({
        text: '',
        startTime: 0,
        maxResults: 1000000
      });

      if (results.length === 0) {
        setStatus('No browsing history found', 'error');
        return;
      }

      // Find min and max years
      const years = new Set();
      results.forEach(item => {
        if (item.lastVisitTime) {
          const year = new Date(item.lastVisitTime).getFullYear();
          years.add(year);
        }
      });

      // Sort years descending (most recent first)
      const sortedYears = Array.from(years).sort((a, b) => b - a);

      // Clear and populate dropdown
      yearSelect.innerHTML = '';
      
      sortedYears.forEach((year, index) => {
        const option = document.createElement('option');
        option.value = year;
        option.textContent = year;
        if (index === 0) option.selected = true;
        yearSelect.appendChild(option);
      });

      // Add test option
      const testOption = document.createElement('option');
      testOption.value = 'test';
      testOption.textContent = 'Last 7 Days (Test)';
      yearSelect.appendChild(testOption);

      setStatus(`Found history from ${sortedYears.length} year(s)`, 'success');
      exportBtn.disabled = false;

    } catch (error) {
      setStatus('Error scanning history: ' + error.message, 'error');
    }
  }

  // Initialize by detecting available years
  await detectAvailableYears();

  exportBtn.addEventListener('click', async () => {
    const selectedValue = yearSelect.value;
    
    setLoading(true);
    setStatus('Fetching history...', 'info');
    statsEl.classList.add('hidden');

    try {
      let startTime, endTime, filename;

      if (selectedValue === 'test') {
        // Test mode: last 7 days
        const testDays = 7;
        endTime = Date.now();
        startTime = endTime - (testDays * 24 * 60 * 60 * 1000);
        filename = `chrome-history-test-${testDays}days.json`;
      } else {
        // Year mode
        const year = parseInt(selectedValue);
        startTime = new Date(`${year}-01-01T00:00:00Z`).getTime();
        endTime = new Date(`${year + 1}-01-01T00:00:00Z`).getTime();
        filename = `chrome-history-${year}.json`;
      }

      // Query history
      const results = await chrome.history.search({
        text: '',
        startTime: startTime,
        endTime: endTime,
        maxResults: 1000000
      });

      if (results.length === 0) {
        setStatus('No history found for this period', 'error');
        setLoading(false);
        return;
      }

      setStatus(`Processing ${results.length} entries...`, 'info');

      // Normalize data
      const normalizedResults = results.map(item => ({
        url: item.url,
        title: item.title || '',
        lastVisitTime: item.lastVisitTime,
        lastVisitDate: new Date(item.lastVisitTime).toISOString(),
        visitCount: item.visitCount,
        typedCount: item.typedCount,
        hostname: extractHostname(item.url)
      }));

      // Sort by most recent first
      normalizedResults.sort((a, b) => b.lastVisitTime - a.lastVisitTime);

      // Create export object
      const exportData = {
        exportDate: new Date().toISOString(),
        dateRange: {
          start: new Date(startTime).toISOString(),
          end: new Date(endTime).toISOString()
        },
        totalEntries: normalizedResults.length,
        history: normalizedResults
      };

      // Create and download JSON
      const jsonString = JSON.stringify(exportData, null, 2);
      const blob = new Blob([jsonString], { type: 'application/json' });
      
      const reader = new FileReader();
      reader.onload = function() {
        chrome.downloads.download({
          url: reader.result,
          filename: filename,
          saveAs: false
        }, (downloadId) => {
          if (chrome.runtime.lastError) {
            setStatus('Download failed: ' + chrome.runtime.lastError.message, 'error');
          } else {
            // Hide button and input group
            document.querySelector('.input-group').style.display = 'none';
            exportBtn.style.display = 'none';
            statusEl.style.display = 'none';
            
            // Show success message
            const successEl = document.getElementById('success-message');
            successEl.classList.remove('hidden');
            
            // Show stats (optional, keeping it for info)
            totalEntriesEl.textContent = normalizedResults.length.toLocaleString();
            statsEl.classList.remove('hidden');
          }
          setLoading(false);
        });
      };
      reader.readAsDataURL(blob);

    } catch (error) {
      setStatus('Error: ' + error.message, 'error');
      setLoading(false);
    }
  });

  function extractHostname(url) {
    try {
      return new URL(url).hostname;
    } catch {
      return '';
    }
  }
});
